//******************************************
// file			: MQ6832_Flash.c
// version		: V1.0
// brief		: flash memory related functions
// note			: make sure that the chip does not power off or reset before the FLASH function is completely
//******************************************   
#include "FlashFunction.h"
#include "MQ6822.h"



//==============================================================================
// name			:Flash_WriteByte 
// retval		: 0:write success  1:write fail<difference from expected value after writing>  2:write fail<not 0xFF before writing>
//            3:invalid address parameter
// param		:<addr>	specified address of flash memory(0x7E80~0x7EFF)
//     		 <data>	one byte of data to be written to flash memory
// note			:make sure the address to be written has no code, so as not to modify the structure of the program after writing
//==============================================================================
unsigned char Flash_WriteByte(unsigned int addr,unsigned char data)
{	
	unsigned char tmpIMF;
	
	
	if( !((addr>=0x7e80&&addr<=0x7eff) ||
		   (addr>=0xc000&&addr<=0xffff)) )
		return 3;				// invalid address
		   
	if( *((unsigned char*)addr) != 0xff)
		return 2;				// not 0xFF before writing
		   
	FLSCR1.byte = 0xa0;		
	FLSCR2.byte = 0xd5;			//enable flash command
	tmpIMF = EIRL.bit.b0;
	EIRL.bit.b0 = 0x0;			//DI
	__asm("ld (0xf555),0xaa");
	__asm("ld (0xfaaa),0x55");
	__asm("ld (0xf555),0xa0");
	*(unsigned char* )(addr) = data;      
	EIRL.bit.b0 = tmpIMF;		//recover IMF
	FLSCR1.byte = 0x40;
	FLSCR2.byte = 0xd5;			//disable flash command
	
	if( *((unsigned char*)addr) != data)
		return 1;				// write fail
	
	
	return 0;					// write success
	
}


//==============================================================================
// name			:Flash_ReadSequence
// retval		:void
// param		:<addr> specified start address of flash memory
//     		 <data> the values of specified addresses of flash memory
//     		 <length>read data length
// note			:
//==============================================================================
void Flash_ReadSequence(unsigned int addr,unsigned char* data,unsigned int length)
{
	unsigned char *p_addr = (unsigned char *)addr;
	unsigned int i;
	for(i=0;i<length;i++)
	{
		data[i] = *(p_addr+i);
		WDCDR.byte=0x4e; 		//clear wdt
	}
	
}




//==============================================================================
// name			:FlashReadByte 
// retval		:unsigned char
// param		:<addr>	specified address of flash memory
// note			:the value of specified address of flash memory
//==============================================================================
unsigned char Flash_ReadByte(unsigned int addr)
{	
	return *((unsigned char *)addr);
}




//==============================================================================
// name			:Flash_SectionErase 
// retval		:0:erase success  1:erase fail 3:invalid address parameter
// param		:<addr> specified address of flash memory area to be erased
// note			:The flash memory area to be erased is specified by the upper 9 bits of address(128 Bytes)
//     
//     		ex. if address =0x7e80 means section:0x7e80~0x7eff including 0x7e80, so erase area is 0x7e80~0x7eff  (128 Bytes)
//         		if address =0x7e99 means section:0x7e80~0x7eff including 0x7e99, so erase area is 0x7e80~0x7eff  (128 Bytes)
//         		if address =0x7eff means section:0x7e80~0x7eff including 0x7eff, so erase area is 0x7e80~0x7eff  (128 Bytes) 
//         		if address =0xf000 means section:0xf000~0xf07f including 0xf000, so erase area is 0xf000~0xf07f  (128 Bytes)
//         		if address =0xcfff means section:0xcf80~0xcfff including 0xcfff, so erase area is 0xcf80~0xcfff  (128 Bytes)
// 
//         section rule:0xXX00~0xXX7f , 0xXX80~0xXXff
// 
// 		   data flash memory area:
//         		<section>0x7e80~0x7eff
//  	   program flash memory area:
//         		<section>0xc000~0xc07f 
//         		<section>0xc080~0xc0ff 
//         		<section>0xc100~0xc17f 
//         		<section>0xc180~0xc1ff
//                         :
//                         :						  
//         		<section>0xff00~0xff7f 
//         		<section>0xff80~0xffff
//==============================================================================
unsigned char Flash_SectionErase(unsigned int addr)
{
	unsigned char tmpIMF;
	unsigned int  iAddr,startAddr,endAddr;
	
	
	if( !((addr>=0x7e80&&addr<=0x7eff) ||
		   (addr>=0xc000&&addr<=0xffff)) )
		return 3;				// invalid address
	
	FLSCR1.byte = 0xa0;		
	FLSCR2.byte = 0xd5;			//enable flash command
	tmpIMF = EIRL.bit.b0;
	EIRL.bit.b0 = 0x0;			//DI
	__asm("ld (0xf555),0xaa");
	__asm("ld (0xfaaa),0x55");
	__asm("ld (0xf555),0x80");
	__asm("ld (0xf555),0xaa");
	__asm("ld (0xfaaa),0x55");		
	*(unsigned char* )(addr) = 0x20;
	EIRL.bit.b0 = tmpIMF;		//recover IMF
	FLSCR1.byte = 0x40;
	FLSCR2.byte = 0xd5;			//disable flash command	
	
	startAddr 	= addr & 0xff80;
	endAddr		= addr|0x7f	;	
	for(iAddr=startAddr;iAddr <=endAddr ;iAddr++){
		if( *(unsigned char* )(iAddr) != 0xff){
			return 1;			// erase fail
		}
	}
	
	return 0;
}
